#ifndef __DST_PARSER_H__
#define __DST_PARSER_H__

#include "DST_Common.h"

/* macros */
#if 0
___Definitions___()
#endif

/*
 * Section syntax macro definitions
 */

#define SECTION_TID           0              /* TID index                    */
#define SECTION_LEN_HI        1              /* length hi index              */
#define SECTION_LEN_LO        2              /* length lo index              */
#define SECTION_TID_EXT_HI	  3              /* Table ID extension high index*/
#define SECTION_TID_EXT_LO    4              /* Table ID extension low index */
#define SECTION_VERSION       5              /* version number index         */
#define SECTION_NUM           6              /* section number index         */
#define SECTION_NUM_LAST      7              /* last section number index    */
#define SECTION_HEADER_LENGTH 8				 /* number of bytes in section header */

#define SECTION_LEN_MASK      0x0FFF         /* length bit mask              */
#define SECTION_VERSION_MASK  0x3E           /* version number bit mask      */
#define SECTION_VERSION_SHIFT 1           	 /* version number bit shift     */
#define SECTION_CNI_MASK      0x01           /* current next indicator mask	 */

#define PID_MASK 			  0x1FFF		 /* Mask for various places a PID is used */
#define PSIMODE_EAGER_FLAG		8

#define MAX_PSI_MPEG_DATA	4096
#define MAX_PSI_SECTIONS	256


#define get_table_id(p)					(p[0])
#define get_section_syntax_indicator(p)	(p[1] >> 7)
#define get_private_indicator(p)		((p[1] >> 6) & 0x1)
#define get_section_length(p)			(((p[1] & 0x0F) << 8) | p[2])
#define get_section_number(p)			(p[6])
#define get_last_section_number(p)		(p[7])

/*
 * Assigned PID's
 */
#define PAT_PID 0x0000
#define EIT_PID 0x0012
#define DIT_PID 0x001E
#define SIT_PID 0x001F

/*
 * Table ID's
 */
#define PROGRAM_ASSOCIATION_SECTION	0
#define CA_SECTION 1
#define TS_PROGRAM_MAP_SECTION 2
#define DIT_SECTION 0x7E
#define SIT_SECTION 0x7F

#define MAX_ECM_SECTION_SIZE	4096
#define TS_ECM_SECTION_ID			0x82
#define TS_EMM_SECTION_ID			0x84

#define COMMON_EMM_TABLE_EXTENSION_ID	0x0000

/***************************** PAT ********************************/

/*
 * MPEG descriptor types (Table 2-39)
 */
#define video_stream_tag 					0x02
#define audio_stream_tag 					0x03
#define hierarchy_tag 						0x04
#define registration_tag 		  			0x05
#define data_stream_alignment_tag 			0x06
#define target_background_grid_tag 			0x07
#define video_window_tag 					0x08
#define CA_tag 								0x09
#define ISO_639_language_tag 				0x0A
#define system_clock_tag 					0x0B
#define multiplex_buffer_utilization_tag 	0x0C
#define copyright_tag 						0x0D
#define maximum_bitrate_tag 				0x0E
#define private_data_indicator_tag 			0x0F
#define smoothing_buffer_tag 				0x10
#define STD_tag 							0x11
#define IBP_tag 							0x12

/* 
 * DVB/ARIB descriptor types 
 */
#define network_name_tag					0x40
#define service_list_tag					0x41
#define DVB_ARIB_stuffing_tag				0x42
#define satellite_delivery_system_tag 		0x43
#define cable_delivery_system_tag			0x44
#define buquet_name_tag						0x47
#define service_tag							0x48
#define country_availability_tag			0x49
#define linkage_tag							0x4A
#define nvod_reference_tag					0x4B
#define DVB_time_shifted_service_tag		0x4C
#define short_event_tag						0x4D
#define extended_event_tag					0x4E
#define time_shifted_event_tag				0x4F
#define component__tag						0x50
#define mosaic_tag							0x51
#define stream_identifier_tag				0x52
#define ca_identifier_tag					0x53
#define content_tag							0x54
#define parental_rating_tag					0x55
#define local_time_offset_tag				0x58
#define partial_transport_stream_tag		0x63

/*
 * ATSC descriptor types
 */
#define ATSC_stuffing_tag					0x80
#define AC3_audio_stream_tag				0x81
#define program_identifier_tag				0x85
#define caption_service_tag					0x86
#define content_advisory_tag				0x87
#define extended_channel_name_tag			0xA0
#define service_location_tag				0xA1
#define ATSC_time_shifted_service_tag 		0xA2
#define component_name_tag					0xA3
#define dcc_departing_request_tag  			0xA8
#define dcc_arriving_request_tag       		0xA9
#define redistribution_control_tag       	0xAA


/*
 * ARIB descriptor types
 */
#define heirarchical_transmission_tag 		0xC0
#define digital_copy_control_tag			0xC1
#define netowkr_identification_tag			0xC2
#define partial_ts_time_tag					0xC3
#define audio_component_tag					0xC4
#define hyperlink_tag						0xC5
#define target_area_tag						0xC6
#define data_control_tag					0xC7
#define video_decode_control_tag			0xC8
#define download_control_tag				0xC9
#define ca_emm_ts_tag						0xCA
#define ca_contrat_information_tag			0xCB
#define ca_service_tag						0xCC
#define basic_local_even_tag				0xD0
#define reference_tag						0xD1
#define node_relation_tag					0xD2
#define shore_node_tag						0xD3
#define stc_reference_tag					0xD4
#define emergency_information_tag			0xFC
#define data_component_tag					0xFD
#define system_management_tag				0xFE


/*
 * MPEG stream types in the PMT (Table 2-29)
 * NOT to be confused with stream ID's found in PES headers
 */
#define StreamType_MPEG1Video	1
#define StreamType_MPEG2Video	2
#define StreamType_MPEG1Audio	3
#define StreamType_MPEG2Audio	4
#define StreamType_MPEG2PrivateSection	5
#define StreamType_MPEG2PESPrivateData	6
#define StreamType_MHEG			7
#define StreamType_AnnexADSMCC	8
#define StreamType_H222_1		9
#define StreamType_DSMCCTypeA	10
#define StreamType_DSMCCTypeB	11
#define StreamType_DSMCCTypeC	12
#define StreamType_DSMCCTypeD	13
#define StreamType_MPEG2Auxiliary 14
#define StreamType_MPEG4Audio	0x11
#define StreamType_MPEG4Video	0x1B
#define StreamType_AACAudio		15
#define StreamType_DC2Video		0x80
#define StreamType_AC3Audio		0x81

typedef enum {
    DVB_TID_network_information_section_actual = 0x40,
    DVB_TID_network_information_section_other  = 0x41,
    DVB_TID_service_description_section_actual = 0x42,
        /* 0x43 ~ 0x45 is reserved. */

    DVB_TID_service_description_section_other  = 0x46,
        /* 0x47 ~ 0x49 is reserved. */

    DVB_TID_bouquet_association_section = 0x4A,
        /* 0x4B ~ 0x4D is reserved. */

    DVB_TID_event_information_section_actual_present = 0x4E,
    DVB_TID_event_information_section_other_present = 0x4F,
    DVB_TID_event_information_section_actual_schedule = 0x50,
        /* 0x50 ~ 0x5F is EIT, actual, schedule */
        
    DVB_TID_event_information_section_other_schedule = 0x60,
        /* 0x60 ~ 0x6F is EIT, other, schedule */

    DVB_TID_time_date_section = 0x70,
    DVB_TID_running_status_section = 0x71,
    DVB_TID_stuffing_section = 0x72,
    DVB_TID_time_offset_section = 0x73,
    DVB_TID_application_information_section = 0x74,
    DVB_TID_container_section = 0x75,
    DVB_TID_related_content_section = 0x76,
    DVB_TID_content_identifier_section = 0x77,
    DVB_TID_MPE_FEC_section = 0x78,
    DVB_TID_resolution_notification_section = 0x79,
        /* 0x79 ~ 0x7D is reserved. */

    DVB_TID_discontinuity_information_section = 0x7E,
    DVB_TID_selection_information_section = 0x7F,

        /* 0x80 ~ 0xFE is user-defined. */
    DVB_TID_broadcaster_information_section = 0xC4,
    DVB_TID_common_data_section = 0xC8,

        /* 0xFF is reserved. */
} DVB_TID_t;


typedef enum {
    /*
     * Mandatory tables for DVB-SI;
     *     NIT(actual), SDT (actual), EIT (actual/present), TDT
     */
    DVB_PID_NIT = 0x10,
    DVB_PID_SDT = 0x11,
    DVB_PID_EIT = 0x12,
    DVB_PID_TDT = 0x14,

    /*
     * Optional tables for DVB-SI;
     *     NIT(other), BAT, SDT (other), EIT (other, actual/schedule), RST, TOT, ST 
     */
    
    DVB_PID_BAT = 0x11,
    DVB_PID_RST = 0x13,
    DVB_PID_TOT = 0x14,

    DVB_PID_BIT	= 0x24,
    DVB_PID_CDT = 0x29,

    ARIB_PID_HEIT = DVB_PID_EIT,
    ARIB_PID_MEIT = 0x26,
    ARIB_PID_LEIT = 0x27,
    
} DVB_PID_t;

#if 0
___Descriptors___()
#endif

typedef enum {
    DVB_TAG_network_name_descriptor = 0x40,
    DVB_TAG_service_list_descriptor = 0x41,
    DVB_TAG_stuffing_descriptor = 0x42,
    DVB_TAG_satellite_delivery_system_descriptor = 0x43,
    DVB_TAG_cable_delivery_system_descriptor = 0x44,
    DVB_TAG_VBI_data_descriptor = 0x45,
    DVB_TAG_VBI_teletext_descriptor = 0x46,
    DVB_TAG_bouquet_name_descriptor = 0x47,
    DVB_TAG_service_descriptor = 0x48,
    DVB_TAG_country_availability_descriptor = 0x49,
    DVB_TAG_linkage_descriptor = 0x4A,
    DVB_TAG_NVOD_reference_descriptor = 0x4B,
    DVB_TAG_time_shifted_service_descriptor = 0x4C,
    DVB_TAG_short_event_descriptor = 0x4D,
    DVB_TAG_extended_event_descriptor = 0x4E,
    DVB_TAG_time_shifted_event_descriptor = 0x4F,
    DVB_TAG_component_descriptor = 0x50,
    DVB_TAG_mosaic_descriptor = 0x51,
    DVB_TAG_stream_identifier_descriptor = 0x52,
    DVB_TAG_CA_identifier_descriptor = 0x53,
    DVB_TAG_content_descriptor = 0x54,
    DVB_TAG_parental_rating_descriptor = 0x55,
    DVB_TAG_teletext_descriptor = 0x56,
    DVB_TAG_telephone_descriptor = 0x57,
    DVB_TAG_local_time_offset_descriptor = 0x58,
    DVB_TAG_subtitling_descriptor = 0x59,
    DVB_TAG_terrestrial_delivery_system_descriptor = 0x5A,
    DVB_TAG_multilingual_network_name_descriptor = 0x5B,
    DVB_TAG_multilingual_bouquet_name_descriptor = 0x5C,
    DVB_TAG_multilingual_service_name_descriptor = 0x5D,
    DVB_TAG_multilingual_component_descriptor = 0x5E,
    DVB_TAG_private_data_specifier_descriptor = 0x5F,
    DVB_TAG_service_move_descriptor = 0x60,
    DVB_TAG_short_smoothing_buffer_descriptor = 0x61,
    DVB_TAG_frequency_list_descriptor = 0x62,
    DVB_TAG_partial_transport_stream_descriptor = 0x63,
    DVB_TAG_data_broadcast_descriptor = 0x64,
    DVB_TAG_scrambling_descriptor = 0x65,
    DVB_TAG_data_broadcast_id_descriptor = 0x66,
    DVB_TAG_transport_stream_descriptor = 0x67,
    DVB_TAG_DSNG_descriptor = 0x68,
    DVB_TAG_PDC_descriptor = 0x69,
    DVB_TAG_AC3_descriptor = 0x6A,
    DVB_TAG_ancillary_data_descriptor = 0x6B,
    DVB_TAG_cell_list_descriptor = 0x6C,
    DVB_TAG_cell_frequency_link_descriptor = 0x6D,
    DVB_TAG_announcement_support_descriptor = 0x6E,
    DVB_TAG_application_signalling_descriptor = 0x6F,
    DVB_TAG_adaptation_field_data_descriptor = 0x70,
    DVB_TAG_service_identifier_descriptor = 0x71,
    DVB_TAG_service_availability_descriptor = 0x72,
    DVB_TAG_default_authority_descriptor = 0x73,
    DVB_TAG_related_content_descriptor = 0x74,
    DVB_TAG_TVA_id_descriptor = 0x75,
    DVB_TAG_content_identifier_descriptor = 0x76,
    DVB_TAG_time_slice_fec_identifier_descriptor = 0x77,
    DVB_TAG_ECM_repetition_rate_descriptor = 0x78,
    DVB_TAG_S2_satellite_delivery_system_descriptor = 0x79,
    DVB_TAG_enhanced_AC3_descriptor = 0x7A,
    DVB_TAG_DTS_descriptor = 0x7B,
    DVB_TAG_AAC_descriptor = 0x7C,
    DVB_TAG_XAIT_location_descriptor = 0x7D,
    DVB_TAG_FTA_content_management_descriptor = 0x7E,
    DVB_TAG_extension_descriptor = 0x7F,
    
    ARIB_TAG_hierarchical_transmission_descriptor       = 0xC0,
    ARIB_TAG_digital_copy_control_descriptor            = 0xC1,
    ARIB_TAG_network_identification_descriptor          = 0xC2,
    ARIB_TAG_partial_transport_stream_time_descriptor   = 0xC3,
    ARIB_TAG_audio_component_descriptor                 = 0xC4,
    ARIB_TAG_hyperlink_descriptor                       = 0xC5,
    ARIB_TAG_target_region_descriptor                   = 0xC6,
    ARIB_TAG_data_content_descriptor                    = 0xC7,
    ARIB_TAG_video_decode_control_descriptor            = 0xC8,
    ARIB_TAG_download_content_descriptor                = 0xC9,
    ARIB_TAG_ca_emm_ts_descriptor                       = 0xCA,
    ARIB_TAG_ca_contract_information_descriptor         = 0xCB,
    ARIB_TAG_ca_service_descriptor                      = 0xCC,
    ARIB_TAG_ts_information_descriptor                  = 0xCD,
    ARIB_TAG_extended_broadcaster_descriptor            = 0xCE,
    ARIB_TAG_logo_transmission_descriptor               = 0xCF,
    ARIB_TAG_basic_local_event_descriptor               = 0xD0,
    ARIB_TAG_reference_descriptor                       = 0xD1,
    ARIB_TAG_node_relation_descriptor                   = 0xD2,
    ARIB_TAG_short_node_information_descriptor          = 0xD3,
    ARIB_TAG_stc_reference_descriptor                   = 0xD4,
    ARIB_TAG_series_descriptor                          = 0xD5,
    ARIB_TAG_event_group_descriptor                     = 0xD6,
    ARIB_TAG_si_parameter_descriptor                    = 0xD7,
    ARIB_TAG_broadcaster_name_descriptor                = 0xD8,
    ARIB_TAG_component_group_descriptor                 = 0xD9,
    ARIB_TAG_si_prime_ts_descriptor                     = 0xDA,
    ARIB_TAG_board_information_descriptor               = 0xDB,
    ARIB_TAG_ldt_linkage_descriptor                     = 0xDC,
    ARIB_TAG_connected_transmission_descriptor          = 0xDD,
    ARIB_TAG_content_availability_descriptor            = 0xDE,
    ARIB_TAG_service_group_descriptor                   = 0xE0,
    ARIB_TAG_carousel_compatible_composite_descriptor   = 0xF7,
    ARIB_TAG_conditional_playback_descriptor            = 0xF8,
    ARIB_TAG_cable_ts_division_system_descriptor        = 0xF9,
    ARIB_TAG_terrestrial_delivery_system_descriptor     = 0xFA,
    ARIB_TAG_partial_reception_descriptor               = 0xFB,
    ARIB_TAG_emergency_information_descriptor           = 0xFC,
    ARIB_TAG_data_component_descriptor                  = 0xFD,
    ARIB_TAG_system_management_descriptor               = 0xFE

} dvb_descriptor_tag_t;


#if 0
___PAT___()
#endif
/*
 * This represents one program within the Program Association Table
 */
typedef struct MPEG_PAT_program {
	DS_U16		program_number;
	DS_U16		program_map_PID;
} MPEG_PAT_program;

/*
 * This contains all interesting fieldsin a Program Association Table.
 * All sections in a multi-section PAT are included in this
 * one structure.
 */
typedef struct pat_t {
	DS_U16				transport_stream_id;
	DS_U8				version_number;
	bool				current_next_indicator;
	DS_U8				section_number;
	DS_U8				last_section_number;
	bool				isWholePAT;				/* FALSE if only a singe section of a PAT */
	DS_U16				numPrograms;
	bool				networkPIDPresent;		/* TRUE if network_pid is valid */
	DS_U16				network_PID;
	MPEG_PAT_program	*programs;
} pat_t, *patPtr_t;


typedef struct pat_t MPEG_PAT;


#if 0
___Network_Information_Table___()
#endif

typedef struct tag_dvb_service_t {
    DS_U16  service_id;
    DS_U8   service_type;
    
} dvb_service_t;

typedef struct tag_dvb_service_list_descriptor_t {
    DS_U16  numServices;
    dvb_service_t *p_service;
    
} dvb_service_list_descriptor_t;

typedef struct tag_arib_transmission_type_t
{
    DS_U8   transmission_type_info;
    DS_U8   number_of_service;
    DS_U16 *service_id;
    
} arib_transmission_type_t;

typedef struct tag_arib_ts_information_descriptor_t
{
    DS_U8   remote_control_key_id;
    
    DS_U8   ts_name_length;
    DS_U8  *ts_name;
    
    DS_U8   transmission_type_count;
    arib_transmission_type_t *transmission_type;
    
} arib_ts_information_descriptor_t;

typedef struct tag_dvb_transport_stream_t {
    DS_U16  transport_stream_id;
    DS_U16  original_network_id;
    DS_U16  transport_descriptor_length;
    DS_U8  *transport_descriptors;
    
    dvb_service_list_descriptor_t *p_service_list;
    arib_ts_information_descriptor_t *p_ts_info;
    
} transport_stream_t;

typedef struct tag_dvb_nit_t {
    DS_U16  network_id;
    DS_U8   version_number;
    DS_U8   section_number;
    DS_U8   last_section_number;
    
    DS_U8   network_name_length;
    DS_U8  *p_network_name;

    DS_U16  system_management_id;
    bool b_system_management_id;
    
    DS_U16  network_descriptor_length;
    DS_U8  *network_descriptors;
    
    DS_U16  num_transport_stream;
    transport_stream_t *transport_streams;
    
} dvb_nit_t;

typedef dvb_nit_t DVB_NIT;


#if 0
___Broadcaster_Information_Table___()
#endif

typedef struct tag_arib_extended_broadcaster_descriptor_t
{
	DS_U8	broadcaster_type;
	DS_U16	terrestrial_broadcaster_id;
	DS_U8	num_affiliation_id;
	DS_U8	num_broadcaster_id;
	DS_U8	*affiliation_id;
	DS_U16	*origin_network_id;
	DS_U8	*broadcaster_id;
} arib_extended_broadcaster_descriptor_t;

typedef struct tag_arib_table_description_t
{
	DS_U8	table_id;
	DS_U8	table_description_length;
	DS_U8	*table_description_byte;
} arib_table_description_t;

typedef struct tag_arib_si_parameter_descriptor_t
{
	DS_U8	parameter_version;
	DS_U16	update_time;
	DS_U16  numTable;
	arib_table_description_t	*table_description;
} arib_si_parameter_descriptor_t;

typedef struct tag_arib_broadcaster_t
{
    DS_U8	broadcaster_id;
    DS_U16  broadcaster_descriptor_length;
	DS_U8   *broadcaster_descriptors;

	arib_extended_broadcaster_descriptor_t *p_ex_broad;
	arib_si_parameter_descriptor_t *p_si_parameter;
	
//   DS_U8   broadcaster_name_length;
//   DS_U8  *p_broadcaster_name;

//	 arib_si_prime_ts_descriptor_t *p_si_prime_ts;
} broadcaster_t;

typedef struct tag_dvb_bit_t
{
    DS_U16  original_network_id;
    DS_U8   version_number;
    DS_U8   section_number;
    DS_U8   last_section_number;

	DS_U16	first_descriptor_length;
	DS_U8	*first_descriptors;

	arib_si_parameter_descriptor_t *p_si_parameter;
//	dvb_service_list_descriptor_t *p_service_list;
//	bool	b_service_list;

	DS_U16  num_broadcaster;
	broadcaster_t *broadcaster;
} dvb_bit_t;

typedef dvb_bit_t DVB_BIT;


#if 0
___Time_Offset_Table___()
#endif

typedef struct tag_dvb_local_time_t {
    DS_U32  country_code;
    DS_U8   country_region_id;
    
    bool local_time_offset_polarity;
    DS_U16  local_time_offset;
    
    DS_U16  time_of_change_date;    /* Date in MJD format. */
    DS_U32  time_of_change_time;    /* Time in BCD format. */
    DS_U16  next_time_offset;
    
} dvb_local_time_t;

typedef struct tag_dvb_local_time_offset_descriptor_t {
    DS_U8   i_num_local_time;
    dvb_local_time_t *p_local_time;
    
} dvb_local_time_offset_descriptor_t;

typedef struct tag_dvb_tot_t {
    DS_U16  date;       /* Current date in MJD format. See annex C of ETSI 300 468. */
    DS_U32  time;       /* Current time in BCD format. 12:35:00 becomes 0x123500 */

    dvb_local_time_offset_descriptor_t *p_lto;

    DS_U16  descriptor_length;
    DS_U8  *descriptors;
} dvb_tot_t;

typedef dvb_tot_t DVB_TOT;


#if 0
___Service_Description_Table___()
#endif

typedef struct tag_dvb_service_descriptor_t {
    DS_U8   service_type;
    
    DS_U8   i_provider_name_length;
    DS_U8  *p_provider_name;
    
    DS_U8   i_service_name_length;
    DS_U8  *p_service_name;
    
} dvb_service_descriptor_t;

typedef struct tag_arib_logo_transmission_descriptor_t
{
	DS_U8	logo_transmission_type;
	DS_U16	logo_id;
	DS_U16	logo_version;
	DS_U16	download_data_id;
	DS_U8	*logo_char;
} arib_logo_transmission_descriptor_t;


typedef struct tag_dvb_sdt_service_t {
    DS_U16  service_id;
    DS_U8   EIT_user_defined_flags;
    bool EIT_schedule_flag;
    bool EIT_present_following_flag;
    DS_U8   running_status;
    bool free_CA_mode;
    
    DS_U16  descriptor_length;
    DS_U8  *descriptors;
    
    dvb_service_descriptor_t *p_service_desc;
    arib_logo_transmission_descriptor_t	*logo_tx_desc;
    
} dvb_sdt_service_t;

typedef struct tag_dvb_sdt_t {
    DS_U16  transport_stream_id;
    DS_U8   version_number;
    DS_U8   section_number;
    DS_U8   last_section_number;
    DS_U16  original_network_id;
    
    DS_U16  numServices;
    dvb_sdt_service_t *services;
    
} dvb_sdt_t;

typedef dvb_sdt_t DVB_SDT;

#if 0
___PMT___()
#endif
/***************************** PMT ********************************/

typedef enum {
	FRAMERATE_23_976    = 1,	/* ATSC */
	FRAMERATE_24        = 2,	/* ATSC */
	FRAMERATE_25        = 3,
	FRAMERATE_29_97     = 4,	/* ATSC */
	FRAMERATE_30        = 5,	/* ATSC */
	FRAMERATE_50        = 6,
	FRAMERATE_59_94     = 7,	/* ATSC */
	FRAMERATE_60        = 8 	/* ATSC */
} MPEG_FRAME_RATE;

typedef enum {
	CHROMA_4_2_0 = 1,		/* ATSC specified */
	CHROMA_4_2_2 = 2,
	CHROMA_4_4_4 = 3
} MPEG_CHROMA_FORMAT;

/* video_stream_descriptor */
typedef struct VideoStreamDescriptor {
	bool				multiple_frame_rate_flag;
	MPEG_FRAME_RATE		frame_rate_code;
	bool				MPEG_1_only_flag;
	bool				constrained_parameter_flag;
	bool				still_picture_flag;
	DS_U8				profile_and_level_indication;
	MPEG_CHROMA_FORMAT 	chroma_format;
	bool				frame_rate_extension_flag;
} VideoStreamDescriptor;

/* Video Decode Control Descriptor (ARIB only) */
typedef struct VideoDecodeControlDescriptor {
	bool				valid;					/* TRUE if this descriptor exists in PMT */
	bool				still_picture_flag;
	bool				sequence_end_code_flag;
	DS_U8				video_decode_format;
} VideoDecodeControlDescriptor;

typedef struct tag_CA_descriptor {
    DS_U16              CA_system_ID;
    DS_U16              CA_PID;
    DS_U8               private_data_length;
    DS_U8              *p_private_data;
} CA_descriptor;

typedef struct MPEG_PMT_stream {
	DS_U8	stream_type;
	DS_U16	elementary_PID;
	DS_U16	descriptor_length;
	DS_U8	*descriptors;
	VideoStreamDescriptor *videoStreamDescriptor;	/* NULL if not present */
	VideoDecodeControlDescriptor videoDecodeControlDescriptor;	/* Not a pointer. Check valid flag */
	
	DS_U8   component_tag;
	bool b_component_tag_valid;
} MPEG_PMT_stream;

/*
 * This contains all interesting fields in a Program Mapping Table
 */
#define DYNARRAY_PMTStream	32		/* Debugger thinks there's this many streams - Does not specify actual length of structure */
typedef struct pmt_t {
//	DS_U16	PID;
	DS_U16	program_number;
	DS_U8	version_number;
	bool	current_next_indicator;
	DS_U16	PCR_PID;
	DS_U16	descriptor_length;
	DS_U8	*descriptors;		/* program descriptors */
	int		numStreams;
	MPEG_PMT_stream	streams[DYNARRAY_PMTStream];
} pmt_t , *pmtPtr_t;

typedef struct pmt_t MPEG_PMT;

#define MPEG_PMTSize(nelem) (sizeof(MPEG_PMT) + \
			((nelem)-DYNARRAY_PMTStream)*sizeof(MPEG_PMT_stream))


#if 0
___Event_Information_Table___()
#endif

#define MAX_SHORT_EVENTS        12
#define MAX_EXTENDED_EVENTS     12
#define MAX_RATINGS             4
#define MAX_AUDIO_COMPONENT     4
#define MAX_CAPTION_ES          8

typedef struct tag_dvb_short_event_descriptor_t {
    DS_U32  ISO_639_language_code;
    DS_U8   event_name_length;
    DS_U8  *p_event_name;
    DS_U8   text_length;
    DS_U8  *p_text;
    
} dvb_short_event_descriptor_t;

typedef struct tag_dvb_ext_event_item_t {
    DS_U8   item_description_length;
    DS_U8  *p_item_description;
    
    DS_U8   item_length;
    DS_U8  *p_item_char;
    
} dvb_ext_event_item_t;

typedef struct tag_dvb_extended_event_descriptor_t {
    DS_U8   descriptor_number;
    DS_U8   last_descriptor_number;
    
    DS_U32  ISO_639_language_code;
    
    DS_U8   numItems;
    dvb_ext_event_item_t *items;
    
    DS_U8   text_length;
    DS_U8  *p_text;
} dvb_extended_event_descriptor_t;

typedef struct tag_dvb_parental_rating_entry_t {
    DS_U32 country_code;
    DS_U8  rating_value;
} dvb_parental_rating_entry_t;


typedef struct tag_dvb_parental_rating_descriptor_t {
    DS_U8   i_num_ratings;
    dvb_parental_rating_entry_t *p_ratings;              /* country_code for MSB 24bits, rating for LSB 8bits. */
    
} dvb_parental_rating_descriptor_t;

typedef struct tag_arib_event_t {
    DS_U16  service_id;
    DS_U16  event_id;
} arib_event_t;

typedef struct tag_arib_other_event_t {
    DS_U16  original_network_id;
    DS_U16  transport_stream_id;
    DS_U16  service_id;
    DS_U16  event_id;
} arib_other_event_t;

typedef struct tag_arib_event_group_descriptor_t {
    DS_U8   group_type;

    DS_U8   numEvents;
    arib_event_t *p_event;
    
    DS_U8   numOtherNetworkEvents;
    arib_other_event_t *p_other_event;
} arib_event_group_descriptor_t;

typedef struct tag_arib_audio_component_descriptor_t
{
	DS_U8	stream_content;
	DS_U8	component_type;
	DS_U8	component_tag;
	DS_U8	stream_type;
	DS_U8	simulcast_group_tag;
	bool	ES_multi_lingual_flag;
	bool	main_component_flag;
	DS_U8	quality_indicator;
	DS_U8	sampling_rate;
	DS_U32	ISO_639_language_code;
	DS_U32	ISO_639_language_code_2;
	DS_U8	*text_char;
} arib_audio_component_descriptor_t;

typedef struct tag_arib_data_content_descriptor_t
{
	DS_U16	data_component_id;
	DS_U8	entry_component;
	DS_U8	selector_length;
	DS_U8	*selector_byte;
	DS_U8	num_of_component_ref;
	DS_U8	*component_ref;
	DS_U32	ISO_639_language_code;
	DS_U8	text_length;
	DS_U8	*text_char;
} arib_data_content_descriptor_t;

typedef struct tag_dvb_eit_event_t {
    DS_U16  event_id;
    DS_U16  start_date;     /* 16-bit LSB of start_time in MJD(Modified Julian Date) format. */
    DS_U32  start_time;     /* 24-bit MSB of start_time in BCD format. */
    DS_U32  duration;       /* BCD format. */
    
    DS_U32  start_time_utc;
    DS_U32  duration_in_sec;
    
    DS_U8   running_status;
    bool free_CA_mode;
    
    int numShortEvents;
    dvb_short_event_descriptor_t **pp_short_event;
    
    int numExtendedEvents;
    dvb_extended_event_descriptor_t **pp_extended_event;
    
    int numParentalRating;
	dvb_parental_rating_descriptor_t **pp_pr_desc;
    
    arib_event_group_descriptor_t *p_event_group;
    
    int numAudioComponent;
    arib_audio_component_descriptor_t **pp_audio_component;
    
    /*
     * Only maintain data_content_descriptor for caption (data_component_id be 0x0008).
     */
    int numDataContent;
    arib_data_content_descriptor_t **pp_data_content;
    
    DS_U16  descriptor_length;
    DS_U8  *descriptors;
} dvb_eit_event_t;

typedef struct tag_dvb_eit_t {
    DS_U8   table_id;
    DS_U16  service_id;
    DS_U8   version_number;
    DS_U8   section_number;
    DS_U8   last_section_number;
    DS_U8   segment_last_section_number;
    
    DS_U16  transport_stream_id;
    DS_U16  original_network_id;
    DS_U8   last_table_id;
    
    DS_U16  numEvents;
    dvb_eit_event_t *eit_events;
    
} dvb_eit_t, *dvb_eitPtr_t;

typedef dvb_eit_t DVB_EIT;
typedef dvb_eitPtr_t P_DVB_EIT;
typedef dvb_eit_event_t DVB_EIT_EVENT;

// TVCT
typedef enum {
	mm_analog			= 0x01,
	mm_SCTE_mode_1		= 0x02,
	mm_SCTE_mode_2		= 0x03,
	mm_ATSC_8VSB		= 0x04,
	mm_ATSC_16VSB		= 0x05,
	/* 0x06-0x7F reserved */
	mm_private_descriptor	= 0x80
	/* 0x81-0xFF private */
} modulation_mode_k;

typedef enum {
	ETM_none			= 0x00,
	ETM_in_this_PTC		= 0x01,
	ETM_in_channel_TSID_PTC	= 0x02
} ETM_location_k;

typedef enum {
	/* 0x00 reserved */
	st_analog_television			= 0x01,
	st_ATSC_digital_television		= 0x02,
	st_ATSC_audio_only			= 0x03,
	st_ATSC_data_broadcast_service	= 0x04
	/* 0x05-0x3F reserved */
} service_type_k;

typedef struct tvctChannel {
	DS_U16			short_name[7];
	DS_U16			major_channel_number;
	DS_U16			minor_channel_number;
	modulation_mode_k		modulation_mode;
	DS_U32			carrier_frequency;
	DS_U16			channel_TSID;
	DS_U16			program_number;
	ETM_location_k		ETM_location;
	bool			access_controlled;
	bool			hidden;
	bool			show_guide;
	service_type_k		service_type;
	DS_U16			source_id;
	DS_U16			descriptor_length;
	DS_U8*			descriptors;		/* un-parsed descriptors */
} tvctChannel_t, *tvctChannelPtr_t;

typedef struct tvctSection {
	DS_U16			transport_stream_id;
	DS_U8				version_number;
	DS_U8				current_next_indicator;
	DS_U8				section_number;
	DS_U8				last_section_number;
	DS_U8				num_channels_in_section;
	tvctChannelPtr_t		channel;
	DS_U16			additional_descriptor_length;
	DS_U8*			additional_descriptors;	/* un-parsed descriptors */
} tvctSection_t, *tvctSectionPtr_t;

typedef struct tvct {
	DS_U16			transport_stream_id;
	DS_U8				version_number;
	DS_U8				numChannels;
	tvctChannelPtr_t		channel;
	DS_U16			additional_descriptor_length;
	DS_U8*			additional_descriptors;	/* un-parsed descriptors */
} tvct_t, *tvctPtr_t;

typedef tvct_t TVCT;

/* Cable Virtual Channel Table (CVCT) */
typedef enum {
	ps_path1		= 0,
	ps_path2		= 1
} path_select_k;

typedef struct cvctChannel {
	DS_U16			short_name[7];
	DS_U16			major_channel_number;
	DS_U16			minor_channel_number;
	modulation_mode_k		modulation_mode;
	DS_U32			carrier_frequency;
	DS_U16			channel_TSID;
	DS_U16			program_number;
	ETM_location_k		ETM_location;
	bool			access_controlled;
	bool			hidden;
	path_select_k		path_select;
	bool			out_of_band;
	bool			show_guide;
	service_type_k		service_type;
	DS_U16			source_id;
	DS_U16			descriptor_length;
	DS_U8*			descriptors;		/* un-parsed descriptors */
} cvctChannel_t, *cvctChannelPtr_t;

typedef struct cvctSection {
	DS_U16			transport_stream_id; /* for oob it is taken as map id */
	DS_U8				version_number;
	DS_U8				current_next_indicator;
	DS_U8				section_number;
	DS_U8				last_section_number;
	DS_U8				num_channels_in_section;
	cvctChannelPtr_t		channel;
	DS_U16			additional_descriptor_length;
	DS_U8*			additional_descriptors;	/* un-parsed descriptors */
} cvctSection_t, *cvctSectionPtr_t;


typedef struct cvct {
	DS_U16			transport_stream_id;
	DS_U8				version_number;
	DS_U8				numChannels;
	cvctChannelPtr_t		channel;
	DS_U16			additional_descriptor_length;
	DS_U8*			additional_descriptors;	/* un-parsed descriptors */
} cvct_t, *cvctPtr_t;

typedef cvct_t CVCT;

// MGT 

typedef enum {
	tt_terrestrial_VCT_cni_1	= 0x0000,
	tt_terrestrial_VCT_cni_0	= 0x0001,
	tt_cable_VCT_cni_1			= 0x0002, /* same for LVCT in OOB */
	tt_cable_VCT_cni_0			= 0x0003, /* same for LVCT in OOB */
	tt_channel_ETT				= 0x0004,
	/* 0x0005-0x00FF Reserved */
	tt_SVCT_VCM                 = 0x0010,   
	tt_SVCT_DCM     		    = 0x0011,   
	tt_SVCT_ICM       			= 0x0012,   
	/* 0x0013 - 0x001F Reserved */
	tt_NIT_CDS					= 0x0020,
	tt_NIT_MMS					= 0x0021,
	/* 0x0022 - 0x002F reserved */
	tt_NTT_SNS					= 0x0030,
	/* 0x0031 - 0x00FF reserved */
	tt_EIT_min				= 0x0100,
	tt_EIT_max				= 0x017F,
	/* 0x0180-0x01FF Reserved */
	tt_event_ETT_min			= 0x0200,
	tt_event_ETT_max			= 0x027F,
	/* 0x0280-0x0300 Reserved */
	tt_RRT_region_1			= 0x0301,
	tt_RRT_region_255			= 0x03FF,
	/* 0x0400-0x0FFF User Private */
	tt_AEIT_min 				= 0x1000,
	tt_AEIT_max					= 0x10FF,
	tt_AETT_min 				= 0x1100,
	tt_AETT_max					= 0x11FF
	/* 0x1200-0xFFFF Reserved */
} table_type_k;

typedef struct mgtTable {
	table_type_k		table_type;
	DS_U16			table_type_PID;
	DS_U8				table_type_version_number;
	DS_U32			number_bytes;
	DS_U16			descriptor_length;
	DS_U8*			descriptors;		/* un-parsed descriptors */
} mgtTable_t, *mgtTablePtr_t;

typedef struct mgtSection {
	DS_U16                  map_id; /* only used in OOB */
	DS_U8				version_number;
	DS_U16			tables_defined;
	mgtTablePtr_t		table;			/* array of size 'tables_defined' */
	DS_U16			descriptor_length;
	DS_U8*			descriptors;		/* un-parsed descriptors */
	DS_U32			CRC32;
} mgtSection_t, *mgtSectionPtr_t;

typedef mgtSection_t MGT;

// EIT
typedef struct eitEvent {
	DS_U16			event_id;
	DS_U32			start_time;
	ETM_location_k		ETM_location;
	DS_U32			length_in_seconds;
	DS_U16			title_length;
	DS_U8				*title;
	DS_U16			descriptor_length;
	DS_U8*			descriptors;		/* un-parsed descriptors */
} eitEvent_t, *eitEventPtr_t;

typedef struct eit {
	DS_U16			source_id;
	DS_U8				version_number;
	DS_U8				numEvents;
	eitEventPtr_t		event;			/* array of size 'numEvents' */
} eit_t, *eitPtr_t;

typedef eit_t	EIT;

// ETT

typedef struct ett {
	DS_U8				version_number;
	DS_U32			ETM_id;
	DS_U16			extended_text_message_length;
	DS_U8				*extended_text_message;
} ettSection_t, *ettSectionPtr_t;

typedef ettSection_t	ETT;


/* Caption Service Descriptor */
typedef enum {
	cct_line21		= 0,
	cct_advanced	= 1
} cc_type_k;

typedef enum {
	l21f_field1		= 0,
	l21f_field2		= 1
} line21_field_k;

typedef struct captionService {
	char			language[4];
	cc_type_k			cc_type;
	union {
		line21_field_k	line21_field;
		DS_U8			caption_service_number;
	} cc_id;
	bool			easy_reader;
	bool			wide_aspect_ratio;
	bool			korean_code;
} captionService_t, *captionServicePtr_t;

typedef struct captionServiceDescriptor {
	DS_U8				number_of_services;
	captionServicePtr_t	service;			/* array of size 'number_of_services' */
} captionServiceDescriptor_t, *captionServiceDescriptorPtr_t;

/* AC-3 Audio Descriptor */
typedef enum {
	sr_48kHz				= 0x0,
	sr_44kHz				= 0x1,
	sr_32kHz				= 0x2,
	/* 0x3 reserved */
	sr_48kHz_or_44kHz			= 0x4,
	sr_48kHz_or_32kHz			= 0x5,
	sr_44kHz_or_32kHz			= 0x6,
	sr_48kHz_or_44kHz_or_32kHz	= 0x7
} sample_rate_code_k;

typedef enum {
	br_32kbps			= 0x00,	br_le_32kbps		= 0x20,
	br_40kbps			= 0x01,	br_le_40kbps		= 0x21,
	br_48kbps			= 0x02,	br_le_48kbps		= 0x22,
	br_56kbps			= 0x03,	br_le_56kbps		= 0x23,
	br_64kbps			= 0x04,	br_le_64kbps		= 0x24,
	br_80kbps			= 0x05,	br_le_80kbps		= 0x25,
	br_96kbps			= 0x06,	br_le_96kbps		= 0x26,
	br_112kbps			= 0x07,	br_le_112kbps		= 0x27,
	br_128kbps			= 0x08,	br_le_128kbps		= 0x28,
	br_160kbps			= 0x09,	br_le_160kbps		= 0x29,
	br_192kbps			= 0x0A,	br_le_192kbps		= 0x2A,
	br_224kbps			= 0x0B,	br_le_224kbps		= 0x2B,
	br_256kbps			= 0x0C,	br_le_256kbps		= 0x2C,
	br_320kbps			= 0x0D,	br_le_320kbps		= 0x2D,
	br_384kbps			= 0x0E,	br_le_384kbps		= 0x2E,
	br_448kbps			= 0x0F,	br_le_448kbps		= 0x2F,
	br_512kbps			= 0x10,	br_le_512kbps		= 0x30,
	br_576kbps			= 0x11,	br_le_576kbps		= 0x31,
	br_640kbps			= 0x12,	br_le_640kbps		= 0x32
} bit_rate_code_k;

typedef enum {
	sm_not_indicated		= 0x0,
	sm_not_Dolby_Surround	= 0x1,
	sm_Dolby_Surround		= 0x2
	/* 0x3 reserved */
} surround_mode_k;

typedef enum {
	bm_main_audio_serv_CM	= 0x0,	/* Complete Main */
	bm_main_audio_serv_ME	= 0x1,	/* Music and Effects */
	bm_associated_serv_VI	= 0x2,	/* Visually Impaired */
	bm_associated_serv_HI	= 0x3,	/* Hearing Impaired */
	bm_associated_serv_D	= 0x4,	/* Dialog */
	bm_associated_serv_C	= 0x5,	/* Commentary */
	bm_associated_serv_E	= 0x6,	/* Emergency */
	bm_as_VO_or_ma_K		= 0x7		/* Voice Over (acmod=001) or Karaoke (acmod=010-111) */
} bit_stream_mode_k;

typedef enum {
	nc_acmod_1_1		= 0x0,
	nc_acmod_1_0		= 0x1,
	nc_acmod_2_0		= 0x2,
	nc_acmod_3_0		= 0x3,
	nc_acmod_2_1		= 0x4,
	nc_acmod_3_1		= 0x5,
	nc_acmod_2_2		= 0x6,
	nc_acmod_3_2		= 0x7,
	nc_encoded_1		= 0x8,
	nc_encoded_le_2		= 0x9,
	nc_encoded_le_3		= 0xA,
	nc_encoded_le_4		= 0xB,
	nc_encoded_le_5		= 0xC,
	nc_encoded_le_6		= 0xD
	/* 0xE-0xF reserved */
} num_channels_k;

typedef enum {
	lc_unknown		= 0x00,	lc_Polish		= 0x20,	lc_background	= 0x40,	lc_Moldavian	= 0x60,
	lc_Albanian		= 0x01,	lc_Portuguese	= 0x21,						lc_Malaysian	= 0x61,
	lc_Breton		= 0x02,	lc_Romanian		= 0x22,						lc_Malagasay	= 0x62,
	lc_Catalan		= 0x03,	lc_Romansh		= 0x23,						lc_Macedonian	= 0x63,
	lc_Croatian		= 0x04,	lc_Serbian		= 0x24,						lc_Laotian		= 0x64,
	lc_Welsh		= 0x05,	lc_Slovak		= 0x25,	lc_Zulu		= 0x45,	lc_Korean		= 0x65,
	lc_Czech		= 0x06,	lc_Slovene		= 0x26,	lc_Vietnamese	= 0x46,	lc_Khmer		= 0x66,
	lc_Danish		= 0x07,	lc_Finnish		= 0x27,	lc_Uzbek		= 0x47,	lc_Kazakh		= 0x67,
	lc_German		= 0x08,	lc_Swedish		= 0x28,	lc_Urdu		= 0x48,	lc_Kannada		= 0x68,
	lc_English		= 0x09,	lc_Turkish		= 0x29,	lc_Ukrainian	= 0x49,	lc_Japanese		= 0x69,
	lc_Spanish		= 0x0A,	lc_Flemish		= 0x2A,	lc_Thai		= 0x4A,	lc_Indonesian	= 0x6A,
	lc_Esperanto	= 0x0B,	lc_Walloon		= 0x2B,	lc_Telugu		= 0x4B,	lc_Hindi		= 0x6B,
	lc_Estonian		= 0x0C,						lc_Tatar		= 0x4C,	lc_Hebrew		= 0x6C,
	lc_Basque		= 0x0D,						lc_Tamil		= 0x4D,	lc_Hausa		= 0x6D,
	lc_Faroese		= 0x0E,						lc_Tadzhik		= 0x4E,	lc_Gurani		= 0x6E,
	lc_French		= 0x0F,						lc_Swahili		= 0x4F,	lc_Gujurati		= 0x6F,
	lc_Frisian		= 0x10,						lc_Sranan_Tongo	= 0x50,	lc_Greek		= 0x70,
	lc_Irish		= 0x11,						lc_Somali		= 0x51,	lc_Georgian		= 0x71,
	lc_Gaelic		= 0x12,						lc_Sinhalese	= 0x52,	lc_Fulani		= 0x72,
	lc_Galician		= 0x13,						lc_Shona		= 0x53,	lc_Dari		= 0x73,
	lc_Icelandic	= 0x14,						lc_Serbo_Croat	= 0x54,	lc_Churash		= 0x74,
	lc_Italian		= 0x15,						lc_Ruthenian	= 0x55,	lc_Chinese		= 0x75,
	lc_Lappish		= 0x16,						lc_Russian		= 0x56,	lc_Burmese		= 0x76,
	lc_Latin		= 0x17,						lc_Quechua		= 0x57,	lc_Bulgarian	= 0x77,
	lc_Latvian		= 0x18,						lc_Pustu		= 0x58,	lc_Bengali		= 0x78,
	lc_Luxembourgian	= 0x19,						lc_Punjabi		= 0x59,	lc_Belorussian	= 0x79,
	lc_Lithuanian	= 0x1A,						lc_Persian		= 0x5A,	lc_Bambora		= 0x7A,
	lc_Hungarian	= 0x1B,						lc_Papamiento	= 0x5B,	lc_Azerbijani	= 0x7B,
	lc_Maltese		= 0x1C,						lc_Oriya		= 0x5C,	lc_Assamese		= 0x7C,
	lc_Dutch		= 0x1D,						lc_Nepali		= 0x5D,	lc_Armenian		= 0x7D,
	lc_Norwegian	= 0x1E,						lc_Ndebele		= 0x5E,	lc_Arabic		= 0x7E,
	lc_Occitan		= 0x1F,						lc_Marathi		= 0x5F,	lc_Amharic		= 0x7F
} language_code_k;

typedef enum {
	tc_UNICODE_2_Byte		= 0,
	tc_ISO8859_1		= 1
} text_code_k;

typedef struct ac3AudioStreamDescriptor {
	sample_rate_code_k	sample_rate_code;
	DS_U8				bsid;
	bit_rate_code_k		bit_rate_code;
	surround_mode_k		surround_mode;
	bit_stream_mode_k		bsmod;
	num_channels_k		num_channels;
	bool			full_svc;
	bool			additional_elements1;	/* indicates that more elements are present */
	language_code_k		langcod;
	language_code_k		langcod2;		/* present if 1+1 mode */
	bool			additional_elements2;	/* indicates that more elements are present */
	union {
		DS_U8			mainid;		
		DS_U8			asvcflags;
	} bsmod_ext;
	DS_U8				textlen;		/* always zero for ATSC */
	text_code_k			text_code;
	DS_U8*			text;
	bool			langflag;			/* ISO 639 language code present */
	bool			langflag_2;			/* 2nd ISO 639 language code for 1+1 */
	char			language[4];			/* ISO 639 code (0 if not defined) */
	char			language2[4];			/* "" */
} ac3AudioStreamDescriptor_t, *ac3AudioStreamDescriptorPtr_t;
		
// Functions

int DHL_PSI_ParsePAT( DS_U8 **sectionArr, MPEG_PAT **returnPat);
int DHL_PSI_ParseDvbSdt(DS_U8 **sectionArr, bool b_actual, dvb_sdt_t **pp_sdt);
void DHL_PSI_FreeMpegSection (void *sectionPtr);
int DHL_PSI_GetMpegDescriptor (DS_U8 *descriptors, DS_U16 len, DS_U8 tag,
							DS_U16 instance, DS_U8 **descriptor);
							
int DHL_PSI_ParseDvbNit(DS_U8 **sectionArr, bool b_actual, dvb_nit_t **pp_nit);							
int DHL_PSI_ParseDvbBit(DS_U8 **sectionArr, dvb_bit_t **pp_bit);			
int DHL_PSI_ParseDvbTot(DS_U8 *sectionArr, dvb_tot_t **pp_tot);
int DHL_PSI_ParseDvbEit(DS_U8 **sectionArr, DVB_EIT **pp_eit);
int DHL_PSI_ParsePMT( DS_U8 *section, MPEG_PMT **returnPmt);
int DHL_PSI_ParseTVCT (DS_U8 **sectionArr, tvctPtr_t *tvctPtr);
int DHL_PSI_ParseCaptionServiceDescriptor (DS_U8* p, captionServiceDescriptorPtr_t *descripPtr);
void DHL_PSI_FreeMpegDescriptor (void *descriptorPtr);
int DHL_PSI_ParseAc3AudioStreamDescriptor (DS_U8* p, ac3AudioStreamDescriptorPtr_t *descripPtr);
int DHL_PSI_ParseMGTSection (DS_U8 *section, mgtSectionPtr_t *mgtSectionPtr);
int DHL_PSI_ParseEIT (DS_U8 **sectionArr, eitPtr_t *eitPtr);
int DHL_PSI_ParseETTSection (DS_U8 *section, ettSectionPtr_t *ettSectionPtr);
int DHL_PSI_ParseCVCT (DS_U8 **sectionArr, cvctPtr_t *cvctPtr);

#endif// __DST_PARSER_H__
